package org.msh.tb.bd.tbforms.dhis2.config;

import org.jboss.seam.annotations.Name;
import org.msh.tb.bd.tbforms.dhis2.DHIS2RequestService;
import org.msh.utils.JsonUtils;
import org.msh.tb.bd.tbforms.dhis2.config.data.ConfigDHIS2Response;
import org.msh.tb.bd.tbforms.dhis2.config.data.DataElementDTO;
import org.msh.tb.bd.tbforms.dhis2.data.AuthResponse;

import java.io.IOException;
import java.io.OutputStream;
import java.net.HttpURLConnection;

/**
 * Created by Mauricio on 06/02/2017.
 * Service that configures a DHIS2 data element to receive information from eTB-Manager
 */
@Name("dhis2DataElementsConfigService")
public class DHIS2DataElementConfigService extends DHIS2RequestService {

    private final String API_URL = "/api/dataElements/[DATA_ELEMENT_ID]";

    /**
     * Send a request to DHIS2 to configure a data element
     * @param credentials
     * @param dataElementId
     * @return
     * @throws IOException
     */
    public void configureDataElement(AuthResponse credentials, String dataElementId) throws IOException {

        if (credentials == null || dataElementId == null || dataElementId.isEmpty()) {
            throw new RuntimeException("Credentials and Data Element Id must not be null.");
        }

        // mount authorization String
        String authorization = "Bearer " + credentials.getAccess_token();

        HttpURLConnection getConn = null;
        HttpURLConnection putConn = null;

        DataElementDTO dataElement;

        try {

            String url = API_URL.replace("[DATA_ELEMENT_ID]", dataElementId);

            // get GET connection
            getConn = super.getGetConnection(url,
                    "fields=aggregationType,categoryCombo,name,shortName,domainType,valueType,zeroIsSignificant",
                    authorization);

            // request
            checkHttpCode(getConn.getResponseCode());

            // read response - response should be the data element required fields to send a PUT request
            dataElement = JsonUtils.parse(getConn.getInputStream(), DataElementDTO.class);

            if (!dataElement.isZeroIsSignificant() || !dataElement.getValueType().equals("INTEGER_ZERO_OR_POSITIVE")) {
                // get PUT connection
                putConn = super.getPutConnection(url, null, authorization);

                // prepare payload
                dataElement.setZeroIsSignificant(true);
                dataElement.setValueType("INTEGER_ZERO_OR_POSITIVE");
                String input = JsonUtils.objectToJSONString(dataElement, false);

                OutputStream os = putConn.getOutputStream();
                os.write(input.getBytes());
                os.flush();

                // request
                checkHttpCode(putConn.getResponseCode());

                // read response - response should be the data element required fields to send a PUT request
                ConfigDHIS2Response res = JsonUtils.parse(putConn.getInputStream(), ConfigDHIS2Response.class);

                if (!res.getStatus().equals("OK")) {
                    throw new RuntimeException("Error configuring data element: " + dataElementId);
                }
            }
        } finally {
            if (getConn != null) {
                getConn.disconnect();
            }
            if (putConn != null) {
                putConn.disconnect();
            }
        }
    }
}
